/*
 * Start with a map file and a list of enclosures and generate am
 * initial database.
 */
#include <unistd.h>
#include <stdio.h>
#include <pthread.h>
#include <errno.h>
#include <netinet/in.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "db.h"
#include "libfma.h"
#include "lf_stdio.h"
#include "lf_fabric.h"
#include "lf_fabric_db.h"
#include "lf_mapper_map.h"
#include "lf_dflt_error.h"
#include "lf_switch.h"
#include "lf_symbols.h"
#include "lf_xbar32.h"
#include "libmyri.h"

#define MAX_MAP_FILES 2		/* accept 2 map files */

/*
 * Struct used to pass args to switch query threads
 */
struct qargs {
  char *name;
  lf_enclosure_t **epp;
  int rc;
};

/*
 * Message to send to tagged xbar to query its ID and incoming port
 */
struct x32_message {
  uint16_t type;
  uint16_t pad;
  struct lf_tagged_xbar_insert tagged_xbar_insert;
};

/*
 * per-xbar private struct used to bond xbars between the 
 * map struct and the fabric strucr
 */
struct xbar_bond {
  struct lf_xbar *map_xbar;	/* mapfile xbar */
  struct lf_xbar *fabric_xbar;	/* fabric xbar */
  int port_delta;	/* how to get from mapfile port numbers to fabric's */

  int rlen;		/* route to this xbar from NIC 0 of this host */
  unsigned char route[32];
  int nic_index;
  int ifc;		/* interface to use from NIC 0 */
  int in_port;		/* port this route beings us to */
  struct lf_xbar *next;	/* used while computing routes */
};

/*
 * per-NIC private struct used to bond nics between the 
 * map struct and the fabric strucr
 */
struct nic_bond {
  struct lf_nic *map_nic;	/* mapfile nic */
  struct lf_nic *fabric_nic;	/* fabric nic */
};

/*
 * global vars
 */
static int Unbound_xbars;
static struct lf_fabric *Fabric;
static int Myri_handle[8];
static int Max_nic;
static int Dflt_low_cnt;

/*
 * local prototypes
 */
static void *run_query(void *);
static struct lf_fabric *load_switches(char *switchfile);
static int load_switch_file(struct lf_fabric *fp, char *switchfile);
static void copy_nic_info(struct lf_mapfile *mp, struct lf_fabric *fp);
static void setup_xbars(struct lf_mapfile *mp, struct lf_fabric *fp);
static int reconcile_fabric(struct lf_mapfile *mp, struct lf_fabric *fp);
static int write_fabric(struct lf_fabric_db *fdp, struct lf_fabric *fp);
static void query_all_enclosures(struct lf_fabric *fp);
static void process_linecards(struct lf_fabric *fp);
static void bond_xbar(struct lf_xbar *mxp, struct lf_xbar *fxp, int delta);
static void connect_xbar_to_nic(struct lf_xbar *fxp, int xport,
                                struct lf_nic *nicp, int nport);
static void connect_xbar_to_xbar(struct lf_xbar *fxp, struct lf_xbar *fxp2,
				int port, int port2);
static void alloc_nic_xcvrs(struct lf_nic *nicp);
static struct lf_host *find_this_host(struct lf_fabric *fp);
static struct lf_xbar *find_xbar_at_route(int nic_index, int ifc,
				unsigned char *route, int rlen, int *port);
static int scan_xbars(struct lf_mapfile *mp, struct lf_fabric *fp);
static void send_invalid_routes(int nic_index, int ifc, unsigned char *route,
			        int rlen, int packets);
static void calc_xbar_routes(struct lf_host *hp, struct lf_mapfile *mp);
static int accrete_map_file(struct lf_mapfile *mp, char *mapfile);
static struct lf_nic *find_nic_by_mac(struct lf_mapfile *mp, lf_mac_addr_t mac);

void
lf_msleep(
  unsigned int msec)
{
  struct timespec t;

  t.tv_sec = msec / 1000;
  t.tv_nsec = (msec % 1000) * 1000000;
  (void) nanosleep(&t, NULL);
}

int
main(
  int argc,
  char **argv)
{
  char *switchfile;
  char *mapfile[2];
  int c;
  struct lf_fabric_db *fdp;
  db_database_ptr_t dbp;
  char *fms_run;
  char *db_name;
  extern char *optarg;
  struct lf_mapfile *mp;
  struct lf_fabric *fp;
  int num_mf;
  int rc;
  int i;

  lf_init();

  /* default args */
  fms_run = getenv(LF_ENV_FMS_RUN);
  if (fms_run == NULL || *fms_run == '\0') fms_run = LF_DFLT_FMS_RUN;
  db_name = getenv(LF_ENV_DB_NAME);
  if (db_name == NULL || *db_name == '\0') db_name = LF_DFLT_DB_NAME;
  num_mf = 0;
  switchfile = NULL;
  Dflt_low_cnt = 500;

  /* parse args list */
  while ((c = getopt(argc, argv, "s:m:H:R:N:l:")) != EOF) switch (c) {

  case 'm':	/* map file (input) */
    if (num_mf >= MAX_MAP_FILES) {
      fprintf(stderr, "Too many map files.\n");
      exit(1);
    }
    mapfile[num_mf++] = optarg;
    break;

  case 's':	/* switch file (input) */
    switchfile = optarg;
    break;

  case 'H':
    fprintf(stderr,
	"Please note: -H has been replaced with -R and is deprecated\n");
    /* FALLSTHROUGH */

  case 'R':	/* wiring list file (output) */
    fms_run = optarg;
    break;

  case 'N':  /* Host port Id to test out of */
    db_name = optarg;
    break;

  case 'l':
    Dflt_low_cnt = atoi(optarg);
    break;
  }

  dbp = db_open_database(fms_run, db_name, TRUE);
  if (dbp == NULL) LF_ERROR(("Error opening database"));

  fdp = lf_fabric_db_handle(dbp);
  if (fdp == NULL) LF_ERROR(("Error getting fabric DB handle"));

  /* open any existing fabric */
  fp = lf_load_fabric(fdp);
  if (fp == NULL) {
    fprintf(stderr, "Please remove existing database at %s/%s and re-run\n",
	fms_run, db_name);
    exit(1);
  }
  if (fp != NULL) lf_free_fabric(fp);

  if (switchfile == NULL) LF_ERROR(("Switch file is required"));
  if (num_mf == 0) LF_ERROR(("Map file is required"));

  /* load the mapper file */
  mp = lf_load_mapper_map(mapfile[0]);
  if (mp == NULL) LF_ERROR(("Loading map file", mapfile[0]));

  /* If additional map files specified, accrete them onto our map */
  for (i=1; i<num_mf; ++i) {
    rc = accrete_map_file(mp, mapfile[i]);
    if (rc == -1) LF_ERROR(("Loading map file", mapfile[i]));
  }

  /* load the switch file to start a fabric */
  fp = load_switches(switchfile);
  if (fp == NULL) LF_ERROR(("Loading switch file %s", switchfile));
  Fabric = fp;

  /* open Myrinet interface */
  Max_nic = -1;
  for (i=0; i<8; ++i) {
    Myri_handle[i] = myri_open(i);
    if (Myri_handle[i] != -1) Max_nic = i;
    if (Myri_handle[i] == -1) {
      if (errno == EBUSY) LF_ERROR(("NIC %d is busy", i));
      if (errno == EPERM) LF_ERROR(("Permission denied opening NIC %d", i));
    }
  }
  if (Max_nic == -1) LF_ERROR(("Error from myri_open"));

  /* do xbar setup */
  setup_xbars(mp, fp);

  /* create NIC instances for everything in the map file */
  copy_nic_info(mp, fp);

  /* reconcile the two files */
  rc = reconcile_fabric(mp, fp);
  if (rc == -1) LF_ERROR(("Error reconciling fabric."));

  /* write out the fabric */
  write_fabric(fdp, fp);
  
  exit(0);

 except:
  exit(1);
}

/*
 * load switch file, creating an enclosure struct for each one.
 */
static int
load_switch_file(
  struct lf_fabric *fp,
  char *switchfile)
{
  FILE *sfp;
  lf_string_t buf;
  char *wp[LF_STRING_LEN];
  int wc;
  struct lf_enclosure *ep;
  int maxenc;

  maxenc = 0;

  sfp = fopen(switchfile, "r");
  if (sfp == NULL) LF_ERROR(("Cannot open %s", switchfile));

  /* read the file */
  while (lf_get_next_line(sfp, buf, wp, &wc) != NULL) {
    if (wc != 1) {
      fprintf(stderr, "Bogus line in switch file containing \"%s\"\n", wp[0]);
      exit(1);
    }

    if (fp->num_enclosures >= maxenc) {
      maxenc += 10;
      fp->enclosures = (struct lf_enclosure **) realloc(fp->enclosures,
				    maxenc * sizeof(struct lf_enclosure *));
      if (fp->enclosures == NULL) LF_ERROR(("realloc enclosures"));
    }

    LF_CALLOC(ep, struct lf_enclosure, 1);
    LF_CALLOC(ep->data, struct lf_enclosure_data, 1);
    LF_DUP_STRING(ep->name, wp[0]);

    fp->enclosures[fp->num_enclosures] = ep;
    ++fp->num_enclosures;
  }

  fclose(sfp);
  return 0;

 except:
  return -1;
}


/*
 * Load the switches file and start creating a fabric from it
 */
static struct lf_fabric *
load_switches(
  char *switchfile)
{
  struct lf_fabric *fp;

  /* allocate the fabric */
  LF_CALLOC(fp, struct lf_fabric, 1);

  /* read the switch file and create enclosure entries */
  load_switch_file(fp, switchfile);

  /* query each switch to get it's type and linecard inventory */
  query_all_enclosures(fp);

  /* add the linecards for each enclosure */
  process_linecards(fp);

  /* we now have a fabric full of enclosures and linecards */
  return fp;

 except:
  return NULL;
}

/*
 * Spawn a seperate thread to query each enclosure in parallel
 */
static void
query_all_enclosures(
  struct lf_fabric *fp)
{
  int i;
  pthread_t *tid;
  struct qargs *qa;
  int rc;

  /* if only 1, no thread - makes debugging easier, too. */
  if (fp->num_enclosures == 1) {
    lf_enclosure_t *ep;
    ep = fp->enclosures[0];
    rc = lf_query_switch(ep->name, &(fp->enclosures[0]));
    if (rc == -1) {
      LF_ERROR(("Error querying %s", ep->name));
    }
    return;
  }

  LF_CALLOC(tid, pthread_t, fp->num_enclosures);
  LF_CALLOC(qa, struct qargs, fp->num_enclosures);

  for (i=0; i<fp->num_enclosures; ++i) {
    qa[i].name = fp->enclosures[i]->name;
    qa[i].epp = fp->enclosures + i;

    rc = pthread_create(tid+i, NULL, run_query, qa+i);
    if (rc != 0) LF_ERROR(("creating thread for %s", fp->enclosures[i]->name));
  }

  for (i=0; i<fp->num_enclosures; ++i) {
    rc = pthread_join(tid[i], NULL);
    if (qa[i].rc != 0) {
       LF_ERROR(("query thread for %s failed", fp->enclosures[i]->name));
    }
  }

  return;

 except:
  exit(1);
}

static void *
run_query(
  void *v)
{
  struct qargs *ap = v;
  ap->rc = lf_query_switch(ap->name, ap->epp);
  pthread_exit(ap);
}


/*
 * Look at the changelist for each enclosure and add appropriate linecards
 */
static void
process_linecards(
  struct lf_fabric *fp)
{
  struct lf_enclosure *ep;
  int i;

  /*
   * Process the change list for each enclosure
   */
  for (i=0; i<fp->num_enclosures; ++i) {
    struct lf_switch_data_change *cp;
    struct lf_switch_data_change *ncp;
    int slot;

    ep = fp->enclosures[i];

    cp = ep->data->change_list;
    ep->data->change_list = NULL;
    while (cp != NULL) {
      ncp = cp->next;

      switch (cp->dc_type) {
	struct lf_linecard *lp;

	case LF_SWITCH_CHANGE_MISSING_LINECARD:
	  slot = cp->c.missing_linecard.slot;
	  printf("%s, slot %d (%s) is missing\n",
	      ep->name, slot, ep->slots[slot]->product_id);
	  break;

	case LF_SWITCH_CHANGE_NEW_LINECARD:
	  slot = cp->c.new_linecard.slot;
	  lp = lf_allocate_linecard(ep, slot, cp->c.new_linecard.product_id,
				    cp->c.new_linecard.serial_no);
	  if (lp == NULL) {
	    LF_ERROR(("Error allocating new linecard for %s, slot %d",
		       ep->name, slot));
	  }
	  printf("%s, new card at slot %d (%s)\n", 
	      ep->name, lf_slot_display_no(lp),
	      cp->c.new_linecard.product_id);
	  break;

	case LF_SWITCH_CHANGE_CHANGED_LINECARD:
	  slot = cp->c.new_linecard.slot;
	  printf("%s, slot %d (%s) changed\n",
	      ep->name, slot, cp->c.new_linecard.product_id);
	  break;
      }
      lf_free_switch_change(cp);
      cp = ncp;
    }
  }
  return;

 except:
  exit(1);
}

/*
 * Copy info about NICs from the mapfile to the fabric struct.
 * This is a little tricky because what we really want is a list of hosts,
 * so we need to try to figure out what host each NIC lives in.  If multiple
 * NICs live inside the same host, we need to consolidate them.
 */
static void
copy_nic_info(
  struct lf_mapfile *mp,
  struct lf_fabric *fp)
{
  int maxhost;
  int ni;
  struct lf_symtab *hsyms;
  int rc;

  maxhost = 0;
  hsyms = lf_symtab_init();

  /* process each NIC in the map file */
  for (ni=0; ni<mp->nnic; ++ni) {
    struct lf_nic *mnp;
    struct lf_nic *fnp;
    struct nic_bond *bp;
    struct lf_host *hp;
    lf_string_t hostname;
    int nic_id;
    int nh;

    rc = 0;
    mnp = mp->nic[ni];		/* get pointer to mapfile NIC */

    /* try to get the hostname by querying it on the fabric */
    for (nh=0; nh<=Max_nic; ++nh) {
      rc = myri_mac_to_hostname(Myri_handle[nh], mnp->mac_addr, hostname,
                                &nic_id);
      if (rc == 0) break;
    }

    /* if it didn't work, use the mac addr as a hostname */
    if (rc != 0) {
      lf_make_mac_hostname(hostname, mnp->mac_addr, mnp->num_ports);
      nic_id = 0;
    }

    /*
     * use the symtab code to make sure we have only one of each host
     * Return of -1 means we've already seen this hostname
     */
    rc = lf_symtab_register(hsyms, hostname);
    if (rc != -1) {

      /* allocate space for more hosts if needed */
      if (fp->num_hosts >= maxhost) {
	maxhost += 50;
	fp->hosts = (struct lf_host **) realloc(fp->hosts,
				      maxhost * sizeof(struct lf_host *));
	if (fp->hosts == NULL) LF_ERROR(("realloc hosts"));
      }

      /* Fill in this host */
      LF_CALLOC(hp, struct lf_host, 1);
      LF_DUP_STRING(hp->hostname, hostname);

      if (rc != fp->num_hosts) {
	fprintf(stderr, "rc speculation failed!\n");
	exit(1);
      }

      /* save host in fabric array */
      fp->hosts[fp->num_hosts] = hp;
      ++fp->num_hosts;

    } else {

      rc = lf_symtab_lookup(hsyms, hostname);
      hp = fp->hosts[rc];

      if (strcmp(hp->hostname, hostname) != 0) {
	fprintf(stderr, "looked up host did not match!\n");
	exit(1);
      }
    }

    /* Now, plug this NIC into its host */
    hp->nics = (struct lf_nic **) realloc(hp->nics,
	              (hp->num_nics+1) * sizeof(struct lf_nic *));
    if (hp->nics == NULL) LF_ERROR(("reallocing NICs"));

    /* allocate and fill in NIC */
    fnp = lf_alloc_nic(mnp->num_ports);
    if (fnp == NULL) LF_ERROR(("Error allocating NIC"));

    memcpy(fnp->mac_addr, mnp->mac_addr, sizeof(fnp->mac_addr));
    LF_DUP_STRING(fnp->product_id, mnp->product_id);
    fnp->slot = hp->num_nics;
    fnp->host_nic_id = nic_id;
    fnp->host = hp;

    alloc_nic_xcvrs(fnp);

    /* bond the mapfile and fabric NICs together */
    LF_CALLOC(bp, struct nic_bond, 1);
    bp->map_nic = mnp;
    bp->fabric_nic = fnp;
    fnp->user.v = bp;
    mnp->user.v = bp;

    /* put it in the host's NIC array */
    hp->nics[hp->num_nics] = fnp;
    ++hp->num_nics;
  }

  lf_symtab_free(hsyms);

  return;

 except:
  exit(1);
}

/*
 * Allocate xcvrs for a NIC
 */
static void
alloc_nic_xcvrs(
  struct lf_nic *nicp)
{
  int p;

  for (p=0; p<nicp->num_ports; ++p) {
    struct lf_xcvr *xcp;

    LF_CALLOC(xcp, struct lf_xcvr, 1);
    xcp->ln_type = LF_NODE_NIC_XCVR;
    xcp->num_conns = 1;
    xcp->num_ports = 2 * xcp->num_conns;

    LF_CALLOC(xcp->ports, union lf_node *, xcp->num_ports);
    LF_CALLOC(xcp->rports, int, xcp->num_ports);

    nicp->phys_ports[p] = LF_NODE(xcp);
    nicp->phys_rports[p] = xcp->num_conns;
    xcp->ports[0 + xcp->num_conns] = LF_NODE(nicp);
    xcp->rports[0 + xcp->num_conns] = p;

    /* fill in parent info */
    xcp->p.nic = nicp;
    xcp->port = p;
  }
  return;

 except:
  exit(1);
}

/*
 * bond two xbars together, then scan through all links known on fabric
 * side and bond those recursively.
 */
static void
bond_xbar(
  struct lf_xbar *mxp,
  struct lf_xbar *fxp,
  int delta)
{
  struct xbar_bond *bp;
  int p;

  /* already bound? make sure it's the same partner */
  bp = mxp->user.v;
  if (bp->fabric_xbar != NULL) {
    if (fxp->user.v == NULL) {
      LF_ERROR(("Inconsistant bond state!"));
    } else if (fxp->user.v != bp) {
      LF_ERROR(("Inconsistant bond partner!"));
    } else {
      return;
    }
  }

  /* make sure port counts are sane relatively - map count may be high */
  if (mxp->num_ports < fxp->num_ports) {
    LF_ERROR(("Port count mismatch"));
  }

  /* check xbar IDs */
  if (fxp->xbar_id > 0) {
    if (mxp->xbar_id == 0) {
      LF_ERROR(("missing map xbar id in bond"));
    } else if (fxp->xbar_id != mxp->xbar_id) {
      LF_ERROR(("xbar ID mismatch in bond"));
    }
  }

  /* bond these two */
  bp->fabric_xbar = fxp;
  bp->port_delta = delta;
  fxp->user.v = bp;

  /* decrement count of unbound xbars */
  --Unbound_xbars;

  /*
   * propagate bond - three scenarios, each handled differently
   * 1) known implicit link on fabric side
   * 2) link to host on mapfile side
   * 3) mapfile link to another bound xbar
   */
  for (p=0; p<fxp->num_ports; ++p) {
    union lf_node *fnp;
    union lf_node *mnp;

    mnp = mxp->topo_ports[p];
    fnp = fxp->topo_ports[p + delta];

    if (mnp != NULL) {

      /* case 1 - implicit link */
      if (fnp != NULL && fnp->ln_type == LF_NODE_XBAR) {
	if (mnp->ln_type != LF_NODE_XBAR) {
	  LF_ERROR(("Node should be xbar"));
	}
	bond_xbar(LF_XBAR(mnp), LF_XBAR(fnp),
		  fxp->topo_rports[p+delta] - mxp->topo_rports[p]);

      /* case 2 - mapfile link to NIC */
      } else if (mnp->ln_type == LF_NODE_NIC) {
	struct lf_nic *mnicp;
	struct lf_nic *fnicp;

	/* find the fabric NIC to connect */
	mnicp = LF_NIC(mxp->topo_ports[p]);
	fnicp = ((struct nic_bond *)(mnicp->user.v))->fabric_nic;
	connect_xbar_to_nic(fxp, p+delta, fnicp, mxp->topo_rports[p]);

      /* case 3 - mapfile shows a link to another xbar */
      } else if (mnp->ln_type == LF_NODE_XBAR) {
	struct lf_xbar *mxp2;
	struct lf_xbar *fxp2;
        struct xbar_bond *bp2;

	mxp2 = LF_XBAR(mnp);
        bp2 = mxp2->user.v;

	/* Is the connected xbar bound ? */
	fxp2 = bp2->fabric_xbar;
	if  (fxp2 != NULL) {

	    /* make the fabric connection */
	  connect_xbar_to_xbar(fxp, fxp2, p+delta,
	      mxp->topo_rports[p]+bp2->port_delta);
	}
      }
    }
  }

  return;

 except:
  exit(1);
}

/*
 * Connecting a crossbar to a NIC - find the corresponding transceiver
 * port for each end and connect them to each other.
 * In the case of a linecard with no transceivers, such as the GigE line
 * cards, just connect the phys ports together instead.
 */
static void
connect_xbar_to_nic(
  struct lf_xbar *fxp,
  int xport,
  struct lf_nic *fnicp,
  int nport)

{
  struct lf_nic *tnp;

  tnp = LF_NIC(fxp->topo_ports[xport]);

  /* make sure fabric xbar is not already connected */
  if (tnp != NULL) {
    if (tnp == fnicp) {
      return;
    } else {
      LF_ERROR(("Bad link in connect_xbar_to_nic"));
    }
  }

  /* make the topo links both ways */
  fxp->topo_ports[xport] = LF_NODE(fnicp);
  fxp->topo_rports[xport] = nport;
  fnicp->topo_ports[nport] = LF_NODE(fxp);
  fnicp->topo_rports[nport] = xport;

  /* Does this linecard have any transceiver ports ?  If it does, expect the
   * NIC to also and connect them together
   */
  if (fxp->linecard->num_xcvrs > 0) {
    struct lf_xcvr *nic_xcp;
    struct lf_xcvr *lc_xcp;
    int nic_xc_port;
    int lc_xc_port;

    /*
     * link the proper xcvrs together - since these two are not already
     * connected (we know because the fabric topo link is NULL) we
     * should check that all the ports we will use are already null.
     */
    nic_xcp = LF_XCVR(fnicp->phys_ports[nport]);
    if (nic_xcp == NULL) LF_ERROR(("NIC has no xcvr!"));
    nic_xc_port = fnicp->phys_rports[nport];
    nic_xc_port -= nic_xcp->num_conns;		/* convert to outbound port */

    lc_xcp = LF_XCVR(fxp->phys_ports[xport]);
    if (lc_xcp == NULL) LF_ERROR(("linecard has no xcvr!"));
    lc_xc_port = fxp->phys_rports[xport];
    lc_xc_port -= lc_xcp->num_conns;		/* convert to outbound port */
  
    if (nic_xcp->ports[nic_xc_port] != NULL) {
      LF_ERROR(("NIC xcvr already connected!"));
    }
    if (lc_xcp->ports[lc_xc_port] != NULL) {
      LF_ERROR(("linecard xcvr already connected!"));
    }

    /* connect them together */
    nic_xcp->ports[nic_xc_port] = LF_NODE(lc_xcp);
    nic_xcp->rports[nic_xc_port] = lc_xc_port;
    lc_xcp->ports[lc_xc_port] = LF_NODE(nic_xcp);
    lc_xcp->rports[lc_xc_port] = nic_xc_port;

  /* This linecard has no transceivers, so assume it is something like
   * a GigE linecard with "resident NICs"
   */
  } else {

    fxp->phys_ports[xport] = LF_NODE(fnicp);
    fxp->phys_rports[xport] = nport;
    fnicp->phys_ports[nport] = LF_NODE(fxp);
    fnicp->phys_rports[nport] = xport;
  }
  return;

 except:
  exit(1);
}

/*
 * Connecting a crossbar to another crossbae - find the corresponding
 * transceiver * port for each end and connect them to each other.
 */
static void
connect_xbar_to_xbar(
  struct lf_xbar *fxp,
  struct lf_xbar *fxp2,
  int port,
  int port2)
{
  struct lf_xcvr *xcp;
  int xc_port;
  struct lf_xcvr *xcp2;
  int xc_port2;

  /* make sure fabric xbars are not already connected */
  if (fxp->topo_ports[port] != NULL) {
    if (fxp->topo_ports[port] == LF_NODE(fxp2)) {
      return;
    } else {
      LF_ERROR(("xbar already linked to inconsistant xbar"));
    }
  }
  if (fxp2->topo_ports[port2] != NULL) {
    if (fxp2->topo_ports[port2] == LF_NODE(fxp)) {
      return;
    } else {
      LF_ERROR(("xbar 2 already linked to inconsistant xbar"));
    }
  }

  /* make the topo links both ways */
  fxp->topo_ports[port] = LF_NODE(fxp2);
  fxp->topo_rports[port] = port2;
  fxp2->topo_ports[port2] = LF_NODE(fxp);
  fxp2->topo_rports[port2] = port;

  /*
   * link the proper xcvrs together - since these two are not already connected
   * (we know because the fabric topo link is NULL) we should check that all 
   * the ports we will use are already null.
   */
  xcp = LF_XCVR(fxp->phys_ports[port]);
  if (xcp == NULL) LF_ERROR(("xbar has no xcvr!"));
  xc_port = fxp->phys_rports[port];
  xc_port -= xcp->num_conns;			/* convert to outbound port */

  xcp2 = LF_XCVR(fxp2->phys_ports[port2]);
  if (xcp2 == NULL) LF_ERROR(("xbar 2 has no xcvr!"));
  xc_port2 = fxp2->phys_rports[port2];
  xc_port2 -= xcp2->num_conns;			/* convert to outbound port */

  if (xcp->ports[xc_port] != NULL) {
    LF_ERROR(("xbar xcvr already connected!"));
  }
  if (xcp2->ports[xc_port2] != NULL) {
    LF_ERROR(("xbar xcvr 2 already connected!"));
  }

  /* connect them together */
  xcp->ports[xc_port] = LF_NODE(xcp2);
  xcp->rports[xc_port] = xc_port2;
  xcp2->ports[xc_port2] = LF_NODE(xcp);
  xcp2->rports[xc_port2] = xc_port;
  return;

 except:
  exit(1);
}

/*
 * Find the port delta for this tagged xbar by sending it a query packet
 * and seeing what absolute port is returned.  Generate a delta by
 * comparing this to what port the map thinks the packet should arrive on.
 */
int
get_x32_in_port(
  struct lf_xbar *mxp,
  int *delta)
{
  struct x32_message msg;
  unsigned char route[64];
  struct xbar_bond*bp;
  int rlen;
  int tries;
  int i;

  bp = mxp->user.v;		/* get to route */

  /* build a route to the xbar and back */
  memcpy(route, bp->route, bp->rlen);
  rlen = bp->rlen;
  route[rlen++] = 0xa0;		/* magic query byte */
  for (i=bp->rlen-1; i>=0; --i) {
    route[rlen++] = LF_DELTA_TO_ROUTE(-LF_ROUTE_TO_DELTA(bp->route[i]));
  }

  /* build a message to send */
  msg.type = 0x1111;
  memset(&msg.tagged_xbar_insert, 0, sizeof(struct lf_tagged_xbar_insert));

  /*
   * Try a couple of times sending the query packet to the xbar
   */
  for (tries=0; tries<5; ++tries) {
    int rc;
    enum myri_event_type event_type;

    rc = myri_raw_send(Myri_handle[bp->nic_index], bp->ifc, route, rlen,
	               &msg, sizeof(msg), NULL);
    if (rc == -1) LF_ERROR(("Error from myri_raw_send"));

    do {
      struct myri_event *mep;

      rc = myri_next_event(Myri_handle[bp->nic_index], &mep, 10);
      if (rc == -1) LF_ERROR(("Error from myri_next_event"));

      event_type = mep->type;
      if (event_type == MYRI_EVENT_RAW_RECV_COMPLETE) {
	struct x32_message *xmp;
	unsigned int rid;

	/* get the returned data */
	xmp = mep->d.raw_recv.rxbuf;
	rid = lf_swap_l(ntohl(xmp->tagged_xbar_insert.id));

	printf("querying xbar %x, port %d, got %x, %d\n",
	    mxp->xbar_id, bp->in_port, rid,
	    xmp->tagged_xbar_insert.absolute_port);

	/* make sure this is from the right tagged xbar */
	if (rid != mxp->xbar_id) {
	  printf("Ignoring tagged xbar reply from %x\n", rid);
	} else {
	  *delta = xmp->tagged_xbar_insert.absolute_port - bp->in_port;
	  myri_release_event(mep);
	  return 0;
	}
      }
      myri_release_event(mep);

    } while (event_type != MYRI_EVENT_NO_EVENT);
  }

 except:
  return -1;
}

/*
 * Reconcile the xbars with xbar_ids between the map file and the fabric.
 */
static void
reconcile_xbar_ids(
  struct lf_mapfile *mp,
  struct lf_fabric *fp)
{
  int x;

  /* loop through all xbars in the map file */
  for (x=0; x<mp->nxbar; ++x) {
    struct lf_xbar *mxp;
    struct lf_xbar *fxp;

    mxp = mp->xbar[x];				/* mapfile xbar */
    fxp = lf_find_xbar_by_id(fp, mxp->xbar_id);	/* matching fabric xbar? */
    if (fxp != NULL) {
      int delta;
      int rc;

      /* get the delta for this xbar */
      rc = get_x32_in_port(mxp, &delta);
      if (rc == -1) LF_ERROR(("Unable to get port delta"));

      bond_xbar(mxp, fxp, delta);	/* if found, bond and propagate */
    }
  }
  return;

 except:
  exit(1);
}

/*
 * Reconcile the map file with the fabric generated from looking at the
 * switches.  First, we'll match up any xbar 32s we have by using their
 * xbar IDs. Then, we'll have to move on to the brute force approach of
 * sending out packets and looking for them in the switch enclosures.
 */
static int
reconcile_fabric(
  struct lf_mapfile *mp,
  struct lf_fabric *fp)
{
  struct lf_host *hp;
  int rc;

  /* find this host - it will be our starting point */
  hp = find_this_host(fp);
  if (hp == NULL) {
    LF_ERROR(("Cannot find self in fabric"));
  }

  /* set routes from this host to all xbars in map file */
  calc_xbar_routes(hp, mp);

  /* take care of all the easy xbars with IDs first */
  reconcile_xbar_ids(mp, fp);

  /* Find all the unbound xbars */
  rc = scan_xbars(mp, fp);
  
  return rc;

 except:
  return -1;
}

/*
 * Calculate routes from this host to all xbars
 */
static void
calc_xbar_routes(
  struct lf_host *hp,
  struct lf_mapfile *mp)
{
  struct lf_xbar *work_list;
  struct lf_xbar *xp;
  struct xbar_bond *bp;
  int n;
  int p;

  /*
   * initialize work_list to all xbars connected to all NICs of
   * this host.
   */
  work_list = NULL;
  for (n=0; n<hp->num_nics; ++n) {
    struct lf_nic *mnp;

    mnp = ((struct nic_bond *)(hp->nics[n]->user.v))->map_nic;

    for (p = 0; p < mnp->num_ports; ++p) {
      xp = LF_XBAR(mnp->topo_ports[p]);

      /* get to private data */
      bp = xp->user.v;
      if (bp->rlen != -1) continue;	/* skip if already set */

      /* fill in route for this xbar */
      bp->nic_index = n;
      bp->ifc = p;
      bp->rlen = 0;
      bp->in_port = mnp->topo_rports[p];

      /* add to work_list */
      bp->next = work_list;
      work_list = xp;
    }
  }

  /*
   * Each loop processes work_list and builds up a new work_list
   * which is one hop farther away.  Then, the new work list is
   * moved to work_list. Repeat loop until work_list is empty. 
   */
  while (work_list != NULL) {

    /* work the list building up routes breadth first */
    xp = work_list;
    work_list = NULL;

    while (xp != NULL) {
      bp = xp->user.v;

      for (p=0; p<xp->num_ports; ++p) {
	struct lf_xbar *xp2;
	struct xbar_bond *bp2;

	xp2 = LF_XBAR(xp->topo_ports[p]);

	/* skip non-xbar or visited connections */
	if (xp2 == NULL || xp2->ln_type != LF_NODE_XBAR) {
	  continue;
	}
	bp2 = xp2->user.v;
	if (bp2->rlen != -1) continue;		/* visited */

	/* fill in route to new xbar */
	memcpy(bp2->route, bp->route, bp->rlen);
	bp2->route[bp->rlen] = LF_DELTA_TO_ROUTE(p - bp->in_port);
	bp2->rlen = bp->rlen+1;
	bp2->nic_index = bp->nic_index;
	bp2->ifc = bp->ifc;
	bp2->in_port = xp->topo_rports[p];

	/* put this on new_work_list */
	bp2->next = work_list;
	work_list = xp2;
      }

      xp = bp->next;		/* next on this list */
    }
  }
}

/*
 * Find this host in the fabric list of hosts.  We do this by
 * finding the NIC matching our NIC's MAC address, then go to its parent
 * to get the host struct
 */
static struct lf_host *
find_this_host(
  struct lf_fabric *fp)
{
  struct myri_nic_info info;
  struct lf_nic *np;
  struct lf_host *hp;
  int rc;
  int n;

  /* get information including MAC address about this NIC */
  rc = myri_get_nic_info(Myri_handle[Max_nic], &info);
  if (rc != 0) LF_ERROR(("Getting NIC info for nic %d", Max_nic));

  /* find matching NIC in fabric */
  np = lf_find_nic_by_mac(fp, info.mac_addr);
  if (np == NULL) LF_ERROR(("Cannot find my NIC %d in fabric", Max_nic));

  /* get parent for this NIC */
  hp = np->host;

  /* check all my NICs according to the fabric opened OK */
  for (n=0; n<hp->num_nics; ++n) {
    if (Myri_handle[hp->nics[n]->host_nic_id] == -1) {
      LF_ERROR(("Nic %d could not be opened...\n", hp->nics[n]->host_nic_id));
    }
  }

  return hp;

 except:
  return NULL;
}

/*
 * See what xbar shows count when we send packets to a particular
 * route.
 */
static struct lf_xbar *
find_xbar_at_route(
  int nic_index,
  int ifc,
  unsigned char *route,
  int rlen,
  int *port)
{
  int tries;
  struct lf_xbar *rxp;
  int rp;
  int low_cnt;
  int high_cnt;
  int inv_cnt;

  /* how many invalid route packets to send and to look for */
  inv_cnt = 500;
  low_cnt = Dflt_low_cnt;
  high_cnt = 1200;

  rxp = NULL;		/* nothing found yet */
  rp = -1;

  /* retry a few times if needed */
  for (tries=0; tries<10; ++tries) {
    int e;

    /* send invalid route packets */
printf("sending packets\n");
    send_invalid_routes(nic_index, ifc, route, rlen, inv_cnt);

    lf_msleep(1500);		/* wait 1 second for switches to clue in */

printf("querying switches\n");
    /* query all enclosures' counters */
    query_all_enclosures(Fabric);

printf("searching for invalid packets\n");
    /* search every enclosure for the packets */
    for (e=0; e<Fabric->num_enclosures; ++e) {
      struct lf_enclosure *ep;
      int l;

      ep = Fabric->enclosures[e];
printf("search %s\n", ep->name);
      for (l=0; l<ep->num_slots; ++l) {
	struct lf_linecard *lp;
	int x;

	lp = ep->slots[l];
	if (lp == NULL) continue;
	for (x=0; x<lp->num_xbars; ++x) {
	  struct lf_xbar *xp;
	  int p;

	  xp = LF_XBAR(lp->xbars[x]);
	  for (p=0; p<xp->num_ports; ++p) {
	    int invrt;

	    /* skip disabled ports */
	    if (xp->data[p].cur_vals.control != 0) continue;

	    invrt = xp->data[p].cur_vals.invalidroutes;
	    invrt -= xp->data[p].old_vals.invalidroutes;

if (invrt > 200) printf("%s %d p%d: %d\n", ep->name, lf_slot_display_no(lp),
p, invrt);
		
	    if (invrt >= low_cnt && invrt <= high_cnt) {

	      /* if return val already set, multiple matches */
	      if (rxp != NULL) {
		fprintf(stderr, "Multiple matches, retry?\n");
		rxp = NULL;
		goto retry;

	      /* otherwise, save this xbar and port */
	      } else {
		rxp = xp;
		rp = p;
	      }
	    }
	  }
	}
      }
    }

    /* if we get here with a return value, all done! */
    if (rxp != NULL) {
      static int conn_cnt;
      ++conn_cnt;
printf("[%d] connection at %s, slot %d, port %d\n", conn_cnt,
    rxp->linecard->enclosure->name, lf_slot_display_no(rxp->linecard), rp);
      *port = rp;
      return rxp;
    }
 retry:
    continue;
  }

  LF_ERROR(("Cannot find xbar"));
 except:
  return NULL;
}

/*
 * Send invalid packets to the specified route
 */
static void
send_invalid_routes(
  int nic_index,
  int ifc,
  unsigned char *route,
  int rlen,
  int packets)
{
  int i;
  unsigned char *byte;
  int rc;

  /* send all the packets */
  byte = 0;
  for (i=0; i<packets; ++i) {
    rc = myri_raw_send(Myri_handle[nic_index], ifc, route, rlen, &byte, 1, NULL);
    if (rc == -1) LF_ERROR(("Error performing raw send"));

    lf_msleep(1);
  }

  /* wait for the send completions */
  while (i > 0) {
    struct myri_event *mep;
    rc = myri_next_event(Myri_handle[nic_index], &mep, -1);
    if (rc == -1)  LF_ERROR(("Error from myri_next_event()"));

    if (mep->type == MYRI_EVENT_RAW_SEND_COMPLETE) {
      --i;
    }
    myri_release_event(mep);
  }

  return;

 except:
  exit(1);
}

/*
 * Scan all xbars outward starting from this xbar.  This is a really
 * straight-forward approach of just knocking off un-bound xbars in a rather
 * arbitrary order and propagating out from each one.
 */
static int
scan_xbars(
  struct lf_mapfile *mp,
  struct lf_fabric *fp)
{
  int xi;
  int num_bound;

  do {
printf("scanning xbars...\n");
    num_bound = 0;		/* none have been bound yet */

    /* loop through xbars looking for unbound ones */
    for (xi=0; xi<mp->nxbar; ++xi) {
      struct lf_xbar *mxp;
      struct xbar_bond *bp;

      mxp = mp->xbar[xi];
      bp = mxp->user.v;

      /* if fabric xbar is not known, find it! */
      if (bp->fabric_xbar == NULL) {
	struct lf_xbar *fxp;
	int port;

        fxp = find_xbar_at_route(bp->nic_index, bp->ifc, bp->route,
			         bp->rlen, &port);

	if (fxp != NULL) {
	  int delta;

	  delta = port - bp->in_port;
	  if (delta >= 0) {
	    bond_xbar(mxp, fxp, delta);
	    ++num_bound;
	  } else {
	    printf("Skipping bogus negative delta\n");
	  }
	}
      }
    }
  } while (num_bound > 0);	/* stop when no more progress */
  return 0;
}

/*
 * Setup all the crossbars.
 * - allocate a "bond" structure for every mapfile xbar
 * - count and compare xbar counts
 */
static void
setup_xbars(
  struct lf_mapfile *mp,
  struct lf_fabric *fp)
{
  int mx;
  int rc;

  /* allocate bond struct for every mapfile xbar */
  for (mx=0; mx<mp->nxbar; ++mx) {
    struct lf_xbar *mxp;
    struct xbar_bond *bp;

    mxp = mp->xbar[mx];
    LF_CALLOC(bp, struct xbar_bond, 1);
    bp->map_xbar = mxp;
    bp->rlen = -1;		/* no route yet */
    mxp->user.v = bp;
  }

  /* generate array of fabric xbars for easy access */
  rc = lf_build_xbar_array(fp);
  if (rc == -1) LF_ERROR(("Error building xbar array"));

  /* compare counts */
  if (mp->nxbar != fp->num_xbars) {
    fprintf(stderr, "Warning: mapfile xbar count (%d) "
		     "!= switch xbar count (%d)\n",
		     mp->nxbar, fp->num_xbars);
    fprintf(stderr, "Mapfile may not be up-to-date.\n");
  }

  return;

 except:
  exit(1);
}

/*
 * Test that dirs are ok before we spend a long time buklding wirelist
 */
void
test_dirs(
  char *fms_run,
  char *db_name)
{
  lf_string_t dir;
  int rc;

  sprintf(dir, "%s/%s", fms_run, db_name);
  rc = mkdir(dir, 0755);
  if (rc != 0 && errno != EEXIST) {
    perror(dir);
    exit(1);
  }
}


/*
 * Write out the fabric database
 */
static int
write_fabric(
  struct lf_fabric_db *fdp,
  struct lf_fabric *fp)
{
  int rc;

  printf("Writing database\n");

  /* clear everything out of any existing database */
  rc = lf_clear_fabric_db(fdp);
  if (rc != 0) LF_ERROR(("Error clearing fabric DB"));

  rc = lf_add_fabric_to_db(fdp, fp);
  if (rc != 0) LF_ERROR(("Error adding fabric to DB"));

  rc = lf_flush_fabric_db(fdp);
  if (rc != 0) LF_ERROR(("Error writing fabric DB"));

  /* close the tables and database */
  lf_close_fabric_db(fdp);

  return 0;

 except:
  return -1;
}

/*
 * Find a NIC in the map file given its MAC address
 */
static struct lf_nic *
find_nic_by_mac(
  struct lf_mapfile *mp,
  lf_mac_addr_t mac)
{
  int n;

  for (n=0; n<mp->nnic; ++n) {
    if (memcmp(mp->nic[n]->mac_addr, mac, sizeof(lf_mac_addr_t)) == 0) {
      return mp->nic[n];
    }
  }
  return NULL;
}

/*
 * Load additional map files and accrete then onto the one we already have
 */
static int
accrete_map_file(
  struct lf_mapfile *mp,
  char *mapfile)
{
  struct lf_mapfile *mp2;
  int old_xbar_cnt;
  int x;
  int n;

  old_xbar_cnt = mp->nxbar;		/* need to save old count */

  /* parse the map file */
  mp2 = lf_load_mapper_map(mapfile);
  if (mp2 == NULL) LF_ERROR(("Loading map file %s", mapfile));

  /*
   * grow the list of xbars, since this map file should be distinct.
   */
  mp->xbar = (struct lf_xbar **)
	  realloc(mp->xbar, (mp->nxbar + mp2->nxbar) * sizeof(struct lf_xbar *));
  if (mp->xbar == NULL) LF_ERROR(("reallocing xbar array"));

  /*
   * copy over all the xbar struct pointers
   */
  for (x=0; x<mp2->nxbar; ++x) {
    mp->xbar[old_xbar_cnt + x] = mp2->xbar[x];
  }
  mp->nxbar += mp2->nxbar;

  /* Now, scan through the array of NICs, expanding each NIC as necessary,
   * copying over the connectivity data, and updating the links in the
   * xbars
   */
  for (n=0; n<mp2->nnic; ++n) {
    struct lf_nic *nic;
    struct lf_nic *nic2;

    nic2 = mp2->nic[n];
    nic = find_nic_by_mac(mp, nic2->mac_addr);

    /* If we cannot find this NIC in the original map, expand the number
     * of NICs and just point to it
     */
    if (nic == NULL) {
      mp->nic = (struct lf_nic **)
	realloc(mp->nic, sizeof(struct lf_nic *) * (mp->nnic+1));
      if (mp->nic == NULL) LF_ERROR(("Reallocing NIC array"));

      mp->nic[mp->nnic] = nic2;
      ++mp->nnic;

    /* If we found a reference to this NIC, expand the number of ports if
     * needed, copy the xbar link info, and update the xbar to point back to
     * the original instance of this NIC
     */
    } else {
      int p;

      /* realloc port array if needed */
      if (nic2->num_ports > nic->num_ports) {
	nic->topo_ports = (union lf_node **)
	  realloc(nic->topo_ports, nic2->num_ports * sizeof(struct ld_node *));
	if (nic->topo_ports == NULL) LF_ERROR(("reallocing topo ports"));

	nic->topo_rports = (int *)
	  realloc(nic->topo_rports, nic2->num_ports * sizeof(int));
	if (nic->topo_rports == NULL) LF_ERROR(("reallocing topo rports"));

	nic->num_ports = nic2->num_ports;
      }

      /* copy over topo links from nic2 */
      for (p=0; p<nic2->num_ports; ++p) {
	if (nic2->topo_ports[p] != NULL) {
	  struct lf_xbar *xp;
	  int xport;

	  /* must not be already connected */
	  if (nic->topo_ports[p] != NULL) {
	    LF_ERROR(("connection specified in multiple map files"));
	  }

	  /* make the link */
	  nic->topo_ports[p] = nic2->topo_ports[p];
	  nic->topo_rports[p] = nic2->topo_rports[p];

	  /* get a pointer to the new location of this xbar */
	  xp = LF_XBAR(nic->topo_ports[p]);

	  /* make the link back to this NIC */
	  xport = nic->topo_rports[p];
	  xp->topo_ports[xport] = LF_NODE(nic);
	}
      }
    }
  }

  return 0;

 except:
  return -1;
}
